%% Fitting a transition model to the benthic d18O data of ODP 846
%
% by Martin H. Trauth, 17 November 2020
% http://mres.uni-potsdam.de

%%
% Mix, AC et al. (1995): Benthic foraminiferal stable isotope stratigraphy
% of site 846: 0-1.8 Ma. In: Pisias, NG; Mayer, LA; Janecek, TR;
% Palmer-Julson, A; van Andel, TH (eds.), Proceedings of the Ocean Drilling
% Program, Scientific Results, College Station, TX (Ocean Drilling
% Program), 138, 839-854, https://doi.org/10.2973/odp.proc.sr.138.160.1995
clear, clc, close all
data1 = load('data_odp846_shackleton_1995.txt');
data2 = load('data_odp846_mix_1995.txt');
data2(:,2) = [];
data = vertcat(data1,data2);
clear data1 data2

%%
data(:,1) = -0.001*data(:,1);
%data = data(data(:,1)>=-4 & data(:,1)<-1,:);

%%
fo = fitoptions('Method','NonlinearLeastSquares',...
    'StartPoint',[0 0 -2.8 1],...
    'Lower',[-20 -10 -5  0],...
    'Upper',[ 20  10  5 10]);
ft = fittype(@(a,b,c,d,x) a+b*(1./(1+exp(-d*(x-c)))),...
    'Coeff',{'a','b','c','d'},...
    'Options',fo);

%%
t = -3.5 : 0.1 : -1;
[f,fgof,fout] = fit(data(:,1),data(:,2),ft)
y = f.a+f.b*(1./(1+exp(-f.d*(t-f.c))));

%%
rft = load('rampfit.dat');
rs = 'c = -2.237 (-2.397,-2.077)';

%%
xl = ['Age (Ma)'];
yl = ['\delta^{18}O (',char(8240),')'];

%%
figure('Position',[50 800 800 250])
axes('Position',[0.1 0.2 0.8 0.7],...
    'XLim',[-7 0],...
    'YLim',[2.5 5],...
    'FontSize',12,...
    'Box','On'), hold on
p = plot(f,data(:,1),data(:,2));
p(1).Color = [0.1 0.3 0.6];
p(1).Marker = 'Square';
p(1).MarkerSize = 6;
p(1).MarkerFaceColor = [0.1 0.3 0.6];
p(1).MarkerEdgeColor = [0.1 0.3 0.6];
p(2).Color = [0.7 0.3 0.1];
p(2).LineWidth = 1.5;
p(2).LineStyle = '-';

line(-rft(:,1),rft(:,3),...                % Ramp Fit
    'Color',[0.7 0.3 0.1],...
    'LineWidth',1.5,...
    'LineStyle','-.')

lgd = legend('Data Points','Sigmoid Fit','Ramp Fit');
lgd.Location = 'northwest';
lgd.Box = 'Off';
lgd.FontSize = 12;
lgd.AutoUpdate = 'Off';
xlabel(xl,...
    'FontSize',12)
ylabel(yl,...
    'FontSize',12)

line([-2.19 -2.19],[4.2 4.6],...              % Inflection Sigmoid Fit
    'LineWidth',1.5,...
    'Color',[0.7 0.3 0.1])
text(-2.19,4.8,'-2.19 Ma',...
    'FontSize',18,...
    'Color',[0.7 0.3 0.1],...
    'HorizontalAlignment','Right')

line([-2.24 -2.24],[2.8 3.2],...              % Inflection Ramp Fit
    'LineWidth',1.5,...
    'LineStyle','-.',...
    'Color',[0.7 0.3 0.1])
text(-2.24,2.65,'-2.24 Ma',...
    'FontSize',18,...
    'Color',[0.7 0.3 0.1],...
    'HorizontalAlignment','Left')

print -dpng -r300 commands_mixshackleton1995_sigmoid_ramp_vs2_FULL.png




